##/*****************************************************************************
## * SIENA: Simulation Investigation for Empirical Network Analysis
## *
## * Web: https://www.stats.ox.ac.uk/~snijders/siena
## *
## * File: multiprint.r
## *
## * Description: This file contains the print and summary modules for the
## * class sienaBayesFit, as well as shortBayesResults.
## * Note: $requestedEffects used everywhere instead of $effects,
## * because the latter also includes all main effects corresponding to
## * included interaction effects,
## * even if these main effects were not requested.
## *
## ****************************************************************************/



##@maketemp Methods
makeTemp <- function(x, groupOnly=0, nfirst, ...)
{
# This reproduces a part of sienaFit but with Bayesian headings;
# for use in print.sienaBayesFit and summary.sienaBayesFit
		name1 <- x$requestedEffects$groupName[1]
		x$requestedEffects <-
			x$requestedEffects[x$requestedEffects$groupName==name1,]
		x$pp <- length(x$theta)
		x$fixed <- x$fixed[x$requestedEffects$groupName==name1]
		if (is.null(nfirst))
		{
			nfirst <- x$nwarm + 1
		}
		tmp <- sienaFitThetaTable(x, fromBayes=TRUE, groupOnly=groupOnly, nfirst=nfirst)
		mydf <- tmp$mydf
		mymat <- as.matrix(mydf[,names(mydf)!="tstat"])
#		mynames <- colnames(mymat)
#		mymat <- cbind(mymat, rep.int(NA, dim(mymat)[1]))
#		mymat <- cbind(mymat, matrix(NA, dim(mymat)[1], 4))
#		colnames(mymat) <- c(mynames, 'random', 'credFrom','credTo', 'p')
		mymat[, 'value'] <-     format(round(mydf$value, digits=4))
		mymat[, 'se'] <-        format(round(mydf$se, digits=4))
		mymat[x$fix, 'se'] <-   "fixed"
		mymat[x$set1, 'random']      <- "   +   "
		mymat[x$set2, 'random']      <- "   -   "
		mymat[x$basicRate, 'random'] <- "       "
		mymat[x$fix, 'random']       <- "       "
		mymat[, 'cFrom'] <- format(round(mydf$cFrom, digits=4))
#		mymat[x$basicRate, 'cFrom'] <- "       "
		mymat[x$fix, 'cFrom']       <- "       "
		mymat[, 'cTo'] <- format(round(mydf$cTo, digits=4))
#		mymat[x$basicRate, 'cTo'] <- "       "
		mymat[x$fix, 'cTo']       <- "       "
		mymat[, 'p'] <- format(round(mydf$p, digits=2))
		mymat[x$fix, 'p'] <-       "       "
		mymat[x$basicRate, 'p'] <- "       "
		if (groupOnly == 0)
		{
			mymat[, 'postSd'] <- format(round(as.numeric(mydf$postSd), digits=4))
			mymat[, 'cSdFrom'] <- format(round(as.numeric(mydf$cSdFrom), digits=4))
			mymat[, 'cSdTo'] <- format(round(as.numeric(mydf$cSdTo), digits=4))
			mymat[(x$fix|x$basicRate|x$set2), 'postSd']    <- "       "
			mymat[(x$fix|x$basicRate|x$set2), 'cSdFrom']   <- "       "
			mymat[(x$fix|x$basicRate|x$set2), 'cSdTo']     <- "       "
		}
		if (x$incidentalBasicRates)
		{
			mymat[x$basicRate, 'se']     <- "       "
#			mymat[x$basicRate, 'postSd'] <- "       "
			mymat[x$basicRate, 'cFrom']  <- "       "
			mymat[x$basicRate, 'cTo']    <- "       "
		}
		mymat[, 'type'] <- format(mymat[, 'type'])
		mymat[, 'text'] <- format(mymat[, 'text'])
		mymat[mydf$row < 1, 'row'] <-
			format(mydf[mydf$row < 1, 'row'])
		mymat[mydf[,'row'] >= 1, 'row'] <-
			paste(format(mydf[mydf$row >= 1, 'row']), '.', sep='')

		if (groupOnly == 0)
		{
			mymat <- rbind(c(rep("", 4), "Post.   ", "", "Post. ", "",
							" cred.  ", " cred. ", " p", "varying  ", "Post.   ", "cred.  ", "cred.  "),
					   c(rep("", 4), "mean    ", "", "s.d.m.", "",
							" from   ", " to    ", "", " ", "s.d.   ","from   ", "to    "),
						mymat)
		}
		else
		{
			mymat <- rbind(c(rep("", 4), "Post.   ", "", "Post. ", "",
							" cred.  ", " cred. ", " p", "varying "),
					   c(rep("", 4), "mean    ", "", "s.d.m.", "",
							" from   ", " to    ", "", ""),
						mymat)
		}
		mymat <- apply(mymat, 2, format)
		tmp1 <- apply(mymat, 1, function(x) paste(x, collapse=" "))
		list(tmp, tmp1)
}

##@print.sienaBayesFit Methods
print.sienaBayesFit <- function(x, nfirst=NULL, ...)
{
	printmat <- function(mat)
	{
		cat(sprintf("%8.4f",mat), sep=c(rep.int(' ', ncol(mat) - 1), '\n'))
	}
	if (!inherits(x, "sienaBayesFit"))
	{
        stop("not a legitimate Siena Bayes model fit")
	}
	else
	{
		cat("Note: this summary does not contain a convergence check.\n")
		if (is.null(nfirst))
		{
			cat("Note: the print function for sienaBayesFit objects")
			cat(" can also use a parameter nfirst,\n")
			cat("      indicating the first run")
			cat(" from which convergence is assumed.\n")
		}
		cat("\n")
		if (length(x$f$groupNames) > 1)
		{
			cat("Groups:\n")
			s <- ""
			ml <- max(nchar(x$f$groupNames))
			fmt <- paste("%-", ml+2, "s", sep="")
			for (i in 1:dim(x$ThinParameters)[2])
			{
				s <- paste(s, gettextf(fmt,x$f$groupNames[i]))
				if ((nchar(s) + ml > 80) | (i == dim(x$ThinParameters)[2]))
				{
					cat(paste(s,"\n"))
					s <- ""
				}
			}
			cat("\n")
		}

		cat("Posterior means and standard deviations ")
		# Make temporary changes to make x look like a sienaFit object
		# so that sienaFitThetaTable can be applied.
		# This is done in function makeTemp.
		if (length(x$f$groupNames) <= 1)
		{
			cat("\n\n")
# For this case, makeTemp still must be adapted.
#			x$theta <- colMeans(x$ThinParameters[,1,])
#			x$covtheta <- cov(x$ThinParameters[,1,])
		}
		else
		{
			cat("for global mean parameters\n\n")
			ntot <- sum(!is.na(x$ThinPosteriorMu[,1]))
			if (is.null(nfirst))
			{
				if (x$frequentist)
				{
					first <- x$nwarm + x$nmain - x$lengthPhase3 + 1
				}
				else
				{
					first <- x$nwarm + 1
				}
			}
			else
			{
				first <- nfirst
			}
			if (first > ntot)
			{
				stop("Not enough data: nfirst too large.")
			}
			cat("Total number of runs in the results is ",ntot, ".\n")
			if (ntot < x$nwarm + x$nmain)
			{
				cat("This object resulted from an intermediate save, after",
					ntot, "MCMC runs.\n\n")
			}
			if (ntot > first+2)
			{
				cat("Posterior means and standard deviations are averages over",
					ntot - first + 1, "MCMC runs (excluding warming, after thinning).\n\n")
#				x$theta <- c(colMeans(x$ThinPosteriorMu[first:ntot,]),
#						colMeans(x$ThinPosteriorEta[first:ntot,, drop=FALSE]))
#				x$covtheta <- cov(cbind(x$ThinPosteriorMu[first:ntot,],
#									x$ThinPosteriorEta[first:ntot,]))
			}
			else
			{
				stop("This object did not come beyond the warming phase.\n")
			}
		}
		tmps <- makeTemp(x, nfirst=nfirst)
		tmp <- tmps[[1]]
		tmp1 <- tmps[[2]]
		addtorow <- tmp$addtorow
		for (i in 1:length(tmp1))
		{
			if (length(addtorow$command) > 0)
			{
				for (j in 1:length(addtorow$command))
				{
					ii <- match(i-1, addtorow$pos[[j]])
					if (!is.na(ii))
						if (i == 2 | addtorow$command[j] == 'Network Dynamics')
							cat( addtorow$command[j], '\n')
						else
							cat('\n', addtorow$command[j], '\n', sep='')
				}
			}
			cat(tmp1[i], '\n')
		}
		if (length(x$f$groupNames) > 1)
		{
			cat("\n")
			if (x$frequentist)
			{
				mean.Sigma <- x$Sigma
				cat("Estimated covariance matrix (varying parameters)\n")
			}
			else
			{
				mean.Sigma <-
					apply(x$ThinPosteriorSigma[first:ntot,,], c(2,3), mean)
				sd.Sigma <-
					apply(x$ThinPosteriorSigma[first:ntot,,], c(2,3), sd)
		cat("Posterior mean of global covariance matrix (varying parameters)\n")
			}
			printmat(mean.Sigma)
			if (!x$frequentist)
			{
				cat("\nPosterior standard deviations of ")
				cat("elements of global covariance matrix\n")
				printmat(sd.Sigma)
			}
		}
		cat("\nFor the rate parameters across all groups:\n")
		rateNames <- x$effectName[x$varyingParametersInGroup][x$ratesInVarying]
		thetaRateMean <- colMeans(x$ThinPosteriorMu[first:ntot,x$ratesInVarying, drop=FALSE],
											na.rm=TRUE)
		thetaRateSD <- apply(x$ThinPosteriorMu[first:ntot,x$ratesInVarying , drop=FALSE],
											2, sd, na.rm=TRUE)
		names(thetaRateMean) <- rateNames
		mydf <- data.frame(Post.mean=thetaRateMean, Post.sd=thetaRateSD)
		rownames(mydf) <- rateNames
		print(round(mydf,5))
#		cat("\nTotal of", ntot-nfirst+1, "samples.\n\n")
		cat("\n")
	}
	invisible(x)
}

##@summary.sienaBayesFit Methods
summary.sienaBayesFit <- function(object, nfirst=NULL, allGroups=FALSE, ...)
{
    if (!inherits(object, "sienaBayesFit"))
	{
        stop("not a legitimate Siena Bayes model fit")
	}
    class(object) <- c("summary.sienaBayesFit", class(object))
	print.summary.sienaBayesFit(object, nfirst, allGroups)
    invisible(object)
}

##@print.summary.sienaBayesFit Methods
print.summary.sienaBayesFit <- function(x, nfirst=NULL, allGroups=FALSE, ...)
{
	if (!inherits(x, "summary.sienaBayesFit"))
	{
		stop("not a legitimate summary of a Siena Bayes model fit")
	}
	ntot <- sum(!is.na(x$ThinPosteriorMu[,1]))
	if (is.null(nfirst))
	{
		if (x$frequentist)
		{
			first <- x$nwarm + x$nmain - x$lengthPhase3 + 1
		}
		else
		{
			first <- x$nwarm + 1
		}
	}
	else
	{
		first <- nfirst
	}
	if (first > ntot)
	{
		stop("Not enough data: nfirst too large.")
	}
	if (x$frequentist)
	{
		cat("Frequentist estimation.\n")
	}
	else
	{
		ncr <- max(sapply(x$effectName, nchar)) + 1
		codestring <- paste("%-", ncr, "s", sep="") # the minus signs leads to right padding
		cat("Bayesian estimation.\n")
		cat("Prior distribution:\n")
		cat("\nMu      ")
		for (i in seq(along=x$priorMu))
		{
			cat(sprintf(codestring, x$effectName[x$varyingParametersInGroup][i]),
				sprintf("%8.4f", x$priorMu[i,1]),"\n        ")
		}
		cat("\nSigma   ")
		for (i in seq(along=x$priorMu))
		{
			cat(sprintf("%8.4f", x$priorSigma[i,]),"\n        ")
		}
		cat("\nPrior Df ",sprintf("%1d", x$priorDf),"\n")
		if (length(x$f$groupNames) >= 2)
		{
			cat("\nKappa  ",sprintf("%8.4f", x$priorKappa),"\n")
		}
		if (!is.null(x$anyPriorEta))
		{
			cat("\nEta   ")
			if (x$anyPriorEta)
			{
				cat("\nFor the fixed parameters, prior mean and variance:\n")
				var.eta <- rep(NA, length(x$set2prior))
				var.eta[x$set2prior] <- 1/(2*x$priorPrecEta)
				if (is.null(x$priorMuEta))
				{
					m.eta <- rep(0, length(x$set2prior))
				}
				else
				{
					m.eta <- x$priorMeanEta
				}

				for (i in seq(along=x$set2prior))
				{
					cat(sprintf(codestring, x$effectName[x$estimatedNonvaryingParametersInGroup][i]))
					if (x$set2prior[i])
					{
						cat(sprintf("%8.4f%8.4f", m.eta[i], var.eta[i]),"\n")
					}
					else
					{
						cat(" (constant prior) \n")
					}
				}
			}
			else
			{
				cat("\nFor the fixed parameters, constant prior.\n")
			}
		}
#		cat("\nFor the basic rate parameters, ")
#		cat("the prior is on the square root scale.\n\n")
	}
	if (x$incidentalBasicRates)
	{
		cat("\nBasic rates parameters are treated as incidental parameters.\n\n")
	}
	cat("\nAlgorithm specifications were ")
	if (!is.null(x$nprewarm))
	{
		cat(" nprewarm =",x$nprewarm,",")
	}
	cat(" nwarm =",x$nwarm,", nmain =", x$nmain,
	    ", nrunMHBatches =", x$nrunMHBatches,
	    ", nImproveMH =", x$nImproveMH,
		",\n nSampVarying =", x$nSampVarying, ", nSampConst =", x$nSampConst,
		", mult =", x$mult, ".\n")
	if (!is.null(nfirst))
	{
		cat("For these results, nwarm is superseded by nfirst = ", nfirst, ".")
	}
	if (ntot < x$nwarm + x$nmain)
	{
		cat("This object resulted from an intermediate save, after",
			ntot, " MCMC runs.")
	}
	if (x$frequentist)
	{
		cat("Lengths of phases were:\n")
		cat("Phase 1:", x$lengthPhase1, ", phase 2:",
			x$nmain - x$lengthPhase3 - x$lengthPhase1,
			", phase 3:", x$lengthPhase1, ".\n")
		cat("Posterior means and standard deviations are")
		cat(" averages over phase 3.\n\n")
	}
	else
	{
		cat("Posterior means and standard deviations are averages over")
		cat(" the last", ntot - first + 1, "runs.\n\n")
	}
	if (ntot > first+2)
	{
		cat("Proportion of acceptances in MCMC proposals after warming up:\n")
		cat(sprintf("%4.2f",
			colMeans(x$ThinBayesAcceptances[first:ntot,])/x$nrunMHBatches),
			fill=TRUE,"\n")
		cat("This should ideally be between 0.15 and 0.50.\n")
	}
	print.sienaBayesFit(x, nfirst)
	if ((ntot > first+2) & allGroups)
	{
		cat("Posterior means and standard deviations of varying parameters per group\n")
		for (h in 1:length(x$f$groupNames))
		{
			cat("\n", x$f$groupNames[h], "\n")
			tmps <- makeTemp(x, groupOnly=h, nfirst=nfirst)
			tmp <- tmps[[1]]
			tmp1 <- tmps[[2]]
			addtorow <- tmp$addtorow
			# first two lines are the header
			lines.thisgroup <-
			   union(c(1,2), 2 + x$ratePositions[[h]])
			lines.thisgroup <- union(lines.thisgroup,
							2 + which(x$varyingObjectiveParameters))
			lines.thisgroup <- union(lines.thisgroup,
							2 + which(x$fix & (!x$basicRate)))
			lines.thisgroup <- sort(union(lines.thisgroup, 2 + which(x$set2)))
			for (i in lines.thisgroup)
			{
				if (length(addtorow$command) > 0)
				{
					for (j in 1:length(addtorow$command))
					{
						ii <- match(i-1, addtorow$pos[[j]])
						if (!is.na(ii))
							if (i == 2 | addtorow$command[j] == 'Network Dynamics')
								cat( addtorow$command[j], '\n')
							else
								cat('\n', addtorow$command[j], '\n', sep='')
					}
				}
				cat(tmp1[i], '\n')
			}
		}
	}
	invisible(x)
}

##@shortBayesResult abbreviated sienaBayesFit results
# also in RSiena. superfluous here.
shortBayesResults <- function(x, nfirst=NULL){
	if (!inherits(x, "sienaBayesFit"))
	{
		stop('x must be a sienaBayesFit object')
	}
	if (is.null(nfirst))
	{
		nfirst <- x$nwarm+1
	}
	df1 <- sienaFitThetaTable(x, fromBayes=TRUE, nfirst=nfirst)[[1]][,
		c("text", "value", "se", "cFrom", "cTo", "postSd", "cSdFrom", "cSdTo" )]
	df1$postSd[is.na(df1$cSdFrom)] <- NA
	df1$postSd <- as.numeric(df1$postSd)
	df1$cSdFrom <- as.numeric(df1$cSdFrom)
	df1$cSdTo <- as.numeric(df1$cSdTo)
	df2 <- as.data.frame(x$requestedEffects[,c("name","shortName", "interaction1", "interaction2",
		"type", "randomEffects", "fix", "parm", "period", "effect1", "effect2", "effect3", "group")])
	df2$period <- as.numeric(df2$period)
	replace1 <- function(x){ifelse(x=="text", "effectName", x)}
	replace2 <- function(x){ifelse(x=="value", "postMeanGlobal", x)}
	replace3 <- function(x){ifelse(x=="se", "postSdGlobal", x)}
	replace4 <- function(x){ifelse(x=="postSd", "postSdBetween", x)}
	dfs <- cbind(df2, df1)
	dfr <- dfs
	names(dfr) <- replace1(replace2(replace3(replace4(names(dfs)))))
	dfr
}

sienaBayes.table <- function(x, nfirst=NULL, d=3,
	filename=paste(deparse(substitute(x)),'.tex',sep=""), align=TRUE)
{
# Produces latex table with summaries of sienaBayesFit object.
# d is number of digits after decimal point.
# This function is in the spirit of RSiena::meta.table
	code <- ifelse(align, "r@{.}l", "c")
	ncols <- ifelse(align, 13, 7)
	sepsign <- ifelse(align, "&", ".")
	numdig <- ifelse(align, 2, 1)
	num2dig <- ifelse(align, 4, 2)
	empty <- ""
	for (i in 1:d){empty <- paste(empty,"")}
	NAstring <- paste(empty, sepsign, empty, sep="")
	startdate <- ifelse(is.null(x$startingDate), "", x$startingDate)
	version <- ifelse(is.null(attr(x,"version")),"",attr(x,"version"))
    fromObjectToLaTeX <- function(a, mw=NULL){
		b <- as.character(a)
		b <- gsub('sqrt', '$\\sqrt{}$', fixed=TRUE, b)
		b <- gsub('->', '$\\rightarrow$', fixed=TRUE, b)
		b <- gsub('<-', '$\\leftarrow$', fixed=TRUE, b)
		b <- gsub('<>', '$\\leftrightarrow$', fixed=TRUE, b)
		b <- gsub('=>', '$\\Rightarrow$', fixed=TRUE, b)
		b <- gsub('>=', '$\\geq$', fixed=TRUE, b)
		b <- gsub('<=', '$\\leq$', fixed=TRUE, b)
# Note: R changes \\ to \ but still displays \\ in printing the string.
		b <- gsub('^(1/1)', '', fixed=TRUE, b)
		b <- gsub('^(1/2)', '$\\sqrt{}$', fixed=TRUE, b)
		b <- gsub('^', '', fixed=TRUE, b)
		b <- gsub('_', '-', fixed=TRUE, b)
		b <- gsub('#', '.', fixed=TRUE, b)
		b <- gsub('&', '.', fixed=TRUE, b)
		format(b, width=mw)
	}
	formatCC <- function(r,h=0){
		number <- formatC(r, digits=d, format="f", decimal.mark=sepsign)
		if (!is.na(r))
		{
			if (r < 0)
			{
				number <- paste("-", number, sep="")
			}
			else
			{
				number <- paste("~~", number, sep="")
			}
		}
		if (h==1)
		{
			number <- paste("(", number, sep="")
		}
		else if (h==2)
		{
			number <- paste(number, ")", sep="")
		}
		ifelse(is.na(r), NAstring, number)
	}
	effs <- x$requestedEffects
	sbr <- multiSiena::shortBayesResults(x, nfirst=nfirst)[!effs$basicRate,]
	efnames <- sbr$effectName
	pmean <- sbr$postMeanGlobal
	cFrom <- sbr$cFrom
	cTo <- sbr$cTo
	psd <- sbr$postSdBetween
	cSdFrom <- sbr$cSdFrom
	cSdTo <- sbr$cSdTo
	neff <- length(pmean)
	max.eff.width <- max(sapply(efnames,
				function(x){nchar(as.character(fromObjectToLaTeX(x)))}))
# header
	line <- c(paste("% Table based on sienaBayes object",
                               deparse(substitute(x))))
	cat("\n",line, "\n", file=filename, append = TRUE)
	line <- c(paste("Estimation date",startdate))
	cat("\n%",line, "\n", file=filename, append = TRUE)
	if (version > "")
	{
		line <- c(paste("Version",version))
		cat("\n%",line, "\n", file=filename, append = TRUE)
	}
# begin table
	cat("\n", sep="", file=filename, append = TRUE)
	line <- "\\begin{tabular}{l "
	for (i in 1:6) {line <- paste(line, code)}
	line <- paste(line,"}")
	cat(line, "\n", sep="", file=filename, append = TRUE)
	cat(" &  \\multicolumn{", numdig, "}{c}{$\\hat\\mu_k , \\hat\\eta_k$} &  \\multicolumn{", numdig, "}{l}{(\\texttt{mFrom}} ",
					file=filename, append = TRUE)
	cat(" & \\multicolumn{", numdig, "}{r}{\\texttt{mTo})} ", file=filename, append = TRUE)
	cat("\n", " &  \\multicolumn{", numdig, "}{c}{$\\hat\\sigma_k$} &  \\multicolumn{", numdig, " }{l}{(\\texttt{sFrom}}",
					" & \\multicolumn{", numdig, "}{r}{\\texttt{sTo})}", file=filename, append = TRUE)
	cat(" \\\\", "\n", sep="", file=filename, append=TRUE)
	cat("\\hline \n", file=filename, append=TRUE)
# body
	for (i in 1:neff)
	{
	line <- paste(fromObjectToLaTeX(efnames[i], max.eff.width), "&")
	{
		line <- paste(line, formatCC(pmean[i]))
		line <- paste(line, " &  ", formatCC(cFrom[i], 1),  sep="")
		line <- paste(line, " &  ",  formatCC(cTo[i], 2),  sep="")
		line <- paste(line, " &  ", formatCC(psd[i]), sep="")
		line <- paste(line, " &  ", formatCC(cSdFrom[i], 1),  sep="")
		line <- paste(line, " &  ",  formatCC(cSdTo[i], 2),  sep="")
	}
	cat(line,"\\\\\n", file=filename, append = TRUE)
	}
# tailer
	cat("\\hline", file=filename, append = TRUE)
	cat("\\multicolumn{", ncols, "}{l}{\\footnotesize \n", file=filename, append = TRUE)
	cat("$\\hat\\mu_k , \\hat\\eta_k$: posterior mean for $\\mu_k$ or $\\eta_k$, as the case may be, }\\\\ \n",
						file=filename, append = TRUE)
	cat("\\multicolumn{", ncols, "}{l}{\\footnotesize \n", file=filename, append = TRUE)
	cat("     with 95\\% credibility interval from \\texttt{mFrom} to \\texttt{mTo};}\\\\ \n",
		  				file=filename, append = TRUE)
	cat("\\multicolumn{", ncols, "}{l}{\\footnotesize \n", file=filename, append = TRUE)
	cat("$\\hat\\sigma_k$: posterior between-groups s.d., }\\\\ \n" , file=filename, append = TRUE)
	cat("\\multicolumn{", ncols, "}{l}{\\footnotesize \n", file=filename, append = TRUE)
	cat("     with 95\\% credibility interval from \\texttt{sFrom} to \\texttt{sTo}.} \n",
						file=filename, append = TRUE)
	cat("\\end{tabular}\n\n", file=filename, append=TRUE)
	if (filename > "")
	{
		cat('Results written (appended) to file', filename,'.\n')
	}
}
